from dataclasses import asdict, dataclass
from typing import Any, DefaultDict, Dict, List, Optional, Tuple

from pyrallis import field


@dataclass
class SAEncoderTrainConfig:
    # wandb params
    project: str = "OSRL-baselines"
    group: str = None
    name: Optional[str] = None
    prefix: Optional[str] = "sa_encoder"
    suffix: Optional[str] = ""
    logdir: Optional[str] = "logs"
    verbose: bool = True
    # dataset params
    outliers_percent: float = None
    noise_scale: float = None
    inpaint_ranges: Tuple[Tuple[float, float], ...] = None
    epsilon: float = None
    density: float = 1.0
    # model params
    state_encode_dim: int = 32
    action_encode_dim: int = 2
    state_encoder_hidden_sizes: List[float] = field(default=[128, 128, 128], is_mutable=True)
    inverse_dynamics_model_hidden_sizes: List[float] = field(default=[128, 128, 128], is_mutable=True)
    action_encoder_hidden_sizes: List[float] = field(default=[128, 128, 128], is_mutable=True)
    idm_loss_weight: float = 0.1
    add_noise: bool = True
    noise_scale: float = 0.1
    # training params
    task: str = "OfflineDroneRun-v0"
    # task: str = "OfflineHopperVelocityGymnasium-v1"
    task_share: str = "Walker2dVelocityGymnasium"
    dataset: str = None
    learning_rate: float = 1e-4
    batch_size: int = 2048
    state_epoch_num: int = 100
    action_epoch_num: int = 100
    steps_per_epoch: int = 1000
    reward_scale: float = 0.1
    cost_scale: float = 1
    num_workers: int = 8
    # evaluation params
    cost_limit: int = 10
    # general params
    seed: int = 0
    device: str = "cuda:1"
    threads: int = 6
    # augmentation param
    deg: int = 4
    pf_sample: bool = False
    beta: float = 1.0
    augment_percent: float = 0.2
    cost_reverse: bool = False
    # maximum absolute value of reward for the augmented trajs
    max_reward: float = 600.0
    # minimum reward above the PF curve
    min_reward: float = 1.0
    # the max drecrease of ret between the associated traj
    # w.r.t the nearest pf traj
    max_rew_decrease: float = 100.0
    # pf only mode param
    pf_only: bool = False
    rmin: float = 300
    cost_bins: int = 60
    npb: int = 5
    cost_sample: bool = False
    linear: bool = True  # linear or inverse
    start_sampling: bool = False
    prob: float = 0.2
    stochastic: bool = True
    init_temperature: float = 0.1
    no_entropy: bool = False
    # random augmentation
    random_aug: float = 0
    aug_rmin: float = 400
    aug_rmax: float = 500
    aug_cmin: float = -2
    aug_cmax: float = 25
    cgap: float = 5
    rstd: float = 1
    cstd: float = 0.2


@dataclass
class SAEncoderCarCircleConfig(SAEncoderTrainConfig):
    pass


@dataclass
class SAEncoderAntRunConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineAntRun-v0"
    # augmentation param
    deg: int = 3
    max_reward: float = 1000.0
    max_rew_decrease: float = 150
    device: str = "cuda:6"


@dataclass
class SAEncoderDroneRunConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 200
    # training params
    task: str = "OfflineDroneRun-v0"
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((400.0, 10), (500.0, 20), (600.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 700.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderDroneCircleConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineDroneCircle-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 1000.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderCarRunConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarRun-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 600.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderAntCircleConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineAntCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderBallRunConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineBallRun-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1400.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderBallCircleConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineBallCircle-v0"
    # augmentation param
    deg: int = 2
    max_reward: float = 1000.0
    max_rew_decrease: float = 200
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarButton1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarButton2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarCircle1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarCircle2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 30.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarGoal1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 50.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarGoal2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderCarPush1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:1"


@dataclass
class SAEncoderCarPush2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineCarPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:7"


@dataclass
class SAEncoderPointButton1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointButton1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 45.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:5"


@dataclass
class SAEncoderPointButton2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointButton2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 50.0
    max_rew_decrease: float = 10
    min_reward: float = 1
    device: str = "cuda:5"


@dataclass
class SAEncoderPointCircle1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointCircle1Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 65.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:5"


@dataclass
class SAEncoderPointCircle2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointCircle2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 55.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:5"


@dataclass
class SAEncoderPointGoal1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointGoal1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"


@dataclass
class SAEncoderPointGoal2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointGoal2Gymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 35.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:4"


@dataclass
class SAEncoderPointPush1Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointPush1Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 20.0
    max_rew_decrease: float = 5
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class SAEncoderPointPush2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflinePointPush2Gymnasium-v0"
    # augmentation param
    deg: int = 0
    max_reward: float = 15.0
    max_rew_decrease: float = 3
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class SAEncoderAntVelocityConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderAntVelocityV0Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderAntVelocityV2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineAntVelocityGymnasium-v2"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderHalfCheetahVelocityConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderHalfCheetahVelocityV0Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderHalfCheetahVelocityV2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHalfCheetahVelocityGymnasium-v2"
    # augmentation param
    deg: int = 1
    max_reward: float = 3000.0
    max_rew_decrease: float = 500
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderHopperVelocityConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderHopperVelocityV0Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderHopperVelocityV2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineHopperVelocityGymnasium-v2"
    # augmentation param
    deg: int = 1
    max_reward: float = 2000.0
    max_rew_decrease: float = 300
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderSwimmerVelocityConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderSwimmerVelocityV0Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderSwimmerVelocityV2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineSwimmerVelocityGymnasium-v2"
    # augmentation param
    deg: int = 1
    max_reward: float = 250.0
    max_rew_decrease: float = 50
    min_reward: float = 1
    device: str = "cuda:6"


@dataclass
class SAEncoderWalker2dVelocityConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v1"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderWalker2dVelocityV0Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v0"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderWalker2dVelocityV2Config(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineWalker2dVelocityGymnasium-v2"
    # augmentation param
    deg: int = 1
    max_reward: float = 3600.0
    max_rew_decrease: float = 800
    min_reward: float = 1
    device: str = "cuda:6"

@dataclass
class SAEncoderEasySparseConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineMetadrive-easysparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class SAEncoderEasyMeanConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easymean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class SAEncoderEasyDenseConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-easydense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 2
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class SAEncoderMediumSparseConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediumsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:3"


@dataclass
class SAEncoderMediumMeanConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-mediummean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (300.0, 20), (300.0, 40))
    # augmentation param
    deg: int = 0
    max_reward: float = 300.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class SAEncoderMediumDenseConfig(SAEncoderTrainConfig):
    # training params
    task: str = "OfflineMetadrive-mediumdense-v0"
    episode_len: int = 1000
    update_steps: int = 200_000


@dataclass
class SAEncoderHardSparseConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardsparse-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class SAEncoderHardMeanConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-hardmean-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


@dataclass
class SAEncoderHardDenseConfig(SAEncoderTrainConfig):
    # model params
    seq_len: int = 20
    episode_len: int = 1000
    # training params
    task: str = "OfflineMetadrive-harddense-v0"
    update_steps: int = 200_000
    target_returns: Tuple[Tuple[float, ...],
                          ...] = ((300.0, 10), (350.0, 20), (400.0, 40))
    # augmentation param
    deg: int = 1
    max_reward: float = 500.0
    max_rew_decrease: float = 100
    min_reward: float = 1
    device: str = "cuda:2"


SAEncoder_DEFAULT_CONFIG = {
    # bullet_safety_gym
    "OfflineCarCircle-v0": SAEncoderCarCircleConfig,
    "OfflineAntRun-v0": SAEncoderAntRunConfig,
    "OfflineDroneRun-v0": SAEncoderDroneRunConfig,
    "OfflineDroneCircle-v0": SAEncoderDroneCircleConfig,
    "OfflineCarRun-v0": SAEncoderCarRunConfig,
    "OfflineAntCircle-v0": SAEncoderAntCircleConfig,
    "OfflineBallCircle-v0": SAEncoderBallCircleConfig,
    "OfflineBallRun-v0": SAEncoderBallRunConfig,
    # safety_gymnasium
    "OfflineCarButton1Gymnasium-v0": SAEncoderCarButton1Config,
    "OfflineCarButton2Gymnasium-v0": SAEncoderCarButton2Config,
    "OfflineCarCircle1Gymnasium-v0": SAEncoderCarCircle1Config,
    "OfflineCarCircle2Gymnasium-v0": SAEncoderCarCircle2Config,
    "OfflineCarGoal1Gymnasium-v0": SAEncoderCarGoal1Config,
    "OfflineCarGoal2Gymnasium-v0": SAEncoderCarGoal2Config,
    "OfflineCarPush1Gymnasium-v0": SAEncoderCarPush1Config,
    "OfflineCarPush2Gymnasium-v0": SAEncoderCarPush2Config,
    # safety_gymnasium: point
    "OfflinePointButton1Gymnasium-v0": SAEncoderPointButton1Config,
    "OfflinePointButton2Gymnasium-v0": SAEncoderPointButton2Config,
    "OfflinePointCircle1Gymnasium-v0": SAEncoderPointCircle1Config,
    "OfflinePointCircle2Gymnasium-v0": SAEncoderPointCircle2Config,
    "OfflinePointGoal1Gymnasium-v0": SAEncoderPointGoal1Config,
    "OfflinePointGoal2Gymnasium-v0": SAEncoderPointGoal2Config,
    "OfflinePointPush1Gymnasium-v0": SAEncoderPointPush1Config,
    "OfflinePointPush2Gymnasium-v0": SAEncoderPointPush2Config,
    # safety_gymnasium: velocity
    "OfflineAntVelocityGymnasium-v1": SAEncoderAntVelocityConfig,
    "OfflineHalfCheetahVelocityGymnasium-v1": SAEncoderHalfCheetahVelocityConfig,
    "OfflineHopperVelocityGymnasium-v1": SAEncoderHopperVelocityConfig,
    "OfflineSwimmerVelocityGymnasium-v1": SAEncoderSwimmerVelocityConfig,
    "OfflineWalker2dVelocityGymnasium-v1": SAEncoderWalker2dVelocityConfig,
    "OfflineAntVelocityGymnasium-v0": SAEncoderAntVelocityV0Config,
    "OfflineHalfCheetahVelocityGymnasium-v0": SAEncoderHalfCheetahVelocityV0Config,
    "OfflineHopperVelocityGymnasium-v0": SAEncoderHopperVelocityV0Config,
    "OfflineSwimmerVelocityGymnasium-v0": SAEncoderSwimmerVelocityV0Config,
    "OfflineWalker2dVelocityGymnasium-v0": SAEncoderWalker2dVelocityV0Config,
    "OfflineAntVelocityGymnasium-v2": SAEncoderAntVelocityV2Config,
    "OfflineHalfCheetahVelocityGymnasium-v2": SAEncoderHalfCheetahVelocityV2Config,
    "OfflineHopperVelocityGymnasium-v2": SAEncoderHopperVelocityV2Config,
    "OfflineSwimmerVelocityGymnasium-v2": SAEncoderSwimmerVelocityV2Config,
    "OfflineWalker2dVelocityGymnasium-v2": SAEncoderWalker2dVelocityV2Config,
    # safe_metadrive
    "OfflineMetadrive-easysparse-v0": SAEncoderEasySparseConfig,
    "OfflineMetadrive-easymean-v0": SAEncoderEasyMeanConfig,
    "OfflineMetadrive-easydense-v0": SAEncoderEasyDenseConfig,
    "OfflineMetadrive-mediumsparse-v0": SAEncoderMediumSparseConfig,
    "OfflineMetadrive-mediummean-v0": SAEncoderMediumMeanConfig,
    "OfflineMetadrive-mediumdense-v0": SAEncoderMediumDenseConfig,
    "OfflineMetadrive-hardsparse-v0": SAEncoderHardSparseConfig,
    "OfflineMetadrive-hardmean-v0": SAEncoderHardMeanConfig,
    "OfflineMetadrive-harddense-v0": SAEncoderHardDenseConfig
}
